%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Function name: apply_cal_corr_conc
%
% Scope:    apply calibration correction to concentrations
%
% Input:    data (structure),
%           bindata (structure),
%           binref (structure),
%           calibration corr index (1-point vs 2-point) (double),
%           cal gas 1 name (string),
%           true values cal gas 1 (double),
%           cal gas 2 name (string),
%           true values cal gas 2 (double),
%           calibration corr index (species) (double),
%           list of gas names (cell),
%           list of gas type "flags" (i.e., MPV positions) (double),
%           index plot requested (logical)
%
% Output:   corrected "bindata" and "binref" structures,
%           overview figures (if requested)
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


function [ref, binref, bindata] = apply_cal_corr_conc(data, bindata, binref, calib_idx_conc, true_gas_conc1, true_conc1, true_gas_conc2, true_conc2, calib_idx_spec, gasnames, gastypeflag, plot_requested)

tic;fprintf('Running function apply_cal_corr_conc...\n');
 
ind_gas1=find(ismember(gasnames,true_gas_conc1));
ind_gas2=find(ismember(gasnames,true_gas_conc2));

target_idx=[];
if calib_idx_spec(1,1)==1
    target_idx(1,width(target_idx)+1)=2;
end

if calib_idx_spec(1,2)==1
    target_idx(1,width(target_idx)+1)=12;
end

if calib_idx_spec(1,3)==1
    target_idx(1,width(target_idx)+1)=13;
end

if calib_idx_spec(1,4)==1
    target_idx(1,width(target_idx)+1)=14;
end

    %% check if CH4 data file was uploaded and modify target_idx accordingly

    if bindata.data_ch4_idx==0
        target_idx = target_idx(find(target_idx~=12 & target_idx~=13 & target_idx~=14));
        fprintf('Warning: No CH4,CO,CO2 data available\n');
    end
    
    %%
    
    if width(target_idx)== 1 || width(target_idx)== 2 || width(target_idx)== 3
        if target_idx(1,1) == 12
           true_conc1(1,1)=true_conc1(1,2); 
           true_conc2(1,1)=true_conc2(1,2); 
        end
        if target_idx(1,1) == 13
           true_conc1(1,1)=true_conc1(1,3); 
           true_conc2(1,1)=true_conc2(1,3); 
        end
        if target_idx(1,1) == 14
           true_conc1(1,1)=true_conc1(1,4); 
           true_conc2(1,1)=true_conc2(1,4); 
        end
    end
    
    if width(target_idx)== 2 || width(target_idx)== 3

        if target_idx(1,2) == 13
           true_conc1(1,2)=true_conc1(1,3); 
           true_conc2(1,2)=true_conc2(1,3); 
        end
        if target_idx(1,2) == 14
           true_conc1(1,2)=true_conc1(1,4); 
           true_conc2(1,2)=true_conc2(1,4); 
        end
    end
    
    if width(target_idx)==3        
        if target_idx(1,3) == 14
           true_conc1(1,3)=true_conc1(1,4); 
           true_conc2(1,3)=true_conc2(1,4); 
        end
    end
        
%% one-point calibration

if calib_idx_conc ==1
    
    % plot data before correction
    if plot_requested
        plot_1point_cal_corr(data,bindata,binref,gasnames,target_idx,'Before calibration correction');
    end
    
    % apply correction
    for z=1:width(target_idx)
        for i=1:width(gastypeflag)
               bindata.(gasnames{1,i})(:,target_idx(1,z))=bindata.(gasnames{1,i})(:,target_idx(1,z))+(true_conc1(1,z)-binref.(gasnames{1,ind_gas1})(1,target_idx(1,z)));
        end
        bindata.all(:,target_idx(1,z))=bindata.all(:,target_idx(1,z))+(true_conc1(1,z)-binref.(gasnames{1,ind_gas1})(1,target_idx(1,z)));    
    end

    % recalculate ref conditions
    [ref, binref, bindata] = calc_ref_conditions(data,bindata,gasnames,gastypeflag);

    % plot data after correction
    if plot_requested
        plot_1point_cal_corr(data,bindata,binref,gasnames,target_idx,'After calibration correction');
    end
end

%% two-point calibration

if calib_idx_conc ==2
   
    % calculate linear fit coefficients true vs measured
    sss=NaN(width(target_idx),2);
    sssplot=NaN(width(target_idx),2);
    for i=1:width(target_idx)
    true=[true_conc1(1,i) true_conc2(1,i)];
    measured=[binref.(gasnames{1,ind_gas1})(1,target_idx(1,i)) binref.(gasnames{1,ind_gas2})(1,target_idx(1,i))];
    [sss(i,:),~]=polyfit(measured,true,1);
    [sssplot(i,:),~]=polyfit(true,measured,1);
    end

    % plot data before correction
    if plot_requested
        plot_2point_cal_corr_conc(data,bindata,binref,gasnames,true_gas_conc1,true_conc1,true_gas_conc2,true_conc2,target_idx,ind_gas1,ind_gas2,sssplot,'Before calibration correction');
    end
    
    % apply correction
    for j=1:width(target_idx)
        for i=1:width(gastypeflag)
               xxx=min(binref.(gasnames{1,ind_gas1})(1,target_idx(1,j)),binref.(gasnames{1,ind_gas2})(1,target_idx(1,j)));
               yyy=min(true_conc1(1,j),true_conc2(1,j));
               bindata.(gasnames{1,i})(:,target_idx(1,j))=sss(j,1)*(bindata.(gasnames{1,i})(:,target_idx(1,j))-xxx)+yyy;     
        end
        bindata.all(:,target_idx(1,j))=sss(j,1)*(bindata.all(:,target_idx(1,j))-xxx)+yyy;
    end

    % recalculate ref conditions
    [ref, binref, bindata] = calc_ref_conditions(data,bindata,gasnames,gastypeflag);

    % recalculate linear fit coefficients true vs measured
    for i=1:width(target_idx)
        true=[true_conc1(1,i) true_conc2(1,i)];
        measured=[binref.(gasnames{1,ind_gas1})(1,target_idx(1,i)) binref.(gasnames{1,ind_gas2})(1,target_idx(1,i))];
        [sss(i,:),~]=polyfit(measured,true,1);
        [sssplot(i,:),~]=polyfit(true,measured,1);
    end

    % plot data after correction
    if plot_requested
        plot_2point_cal_corr_conc(data,bindata,binref,gasnames,true_gas_conc1,true_conc1,true_gas_conc2,true_conc2,target_idx,ind_gas1,ind_gas2,sssplot,'After calibration correction');
    end     
end

%%
time_elapsed=toc; fprintf('apply_cal_corr_conc completed (execution time: %1.2f s)\n',time_elapsed); 

end

